import LicenseSpring

import SwiftUI

struct LicenseGeneralInfoView: View {
    let manager: LicenseManaging?
    let license: License?
    
    @State private var consumptionDelta = 0
    @State private var borrowFor = 0
    @State private var borrowMultiplier = 1
    @State private var productDetails: ProductDetails?
    @State private var isAirGapDeativationShown = false
    @State private var selectedFeature: LicenseFeature?
    private let minButtonWidth: CGFloat = 140
    
    @EnvironmentObject private var viewUpdater: ViewUpdaterHack
    
    var body: some View {
        HStack {
            VStack(alignment: .leading) {
                MinWidthButton("Product Details", width: minButtonWidth) { productDetails = license?.productDetails }
                Text("License file: \("")")
                HStack {
                    MinWidthButton("Open License Folder...", width: minButtonWidth) {
                        guard let location = license?.configuration.licenseFile else { return }
                        NSWorkspace.shared.activateFileViewerSelecting([location])
                    }
                    MinWidthButton("Clear License Folder", width: minButtonWidth) {
                        performAction(name: "Clear License Folder") {
                            try manager?.clearLocalStorage()
                        }
                    }
                }
                HStack {
                    VStack {
                        MinWidthButton("Check License sync", width: minButtonWidth) {
                            performAction(name: "Check License") {
                                try license?.fullCheck()
                            }
                        }
                        MinWidthButton("Check License async", width: minButtonWidth) {
                            performAction(name: "Check License") {
                                license?.fullCheckAsync(env: "mac") { error in
                                    if let error {
                                        showAlert(title: "Error",
                                                  text: error.localizedDescription,
                                                  style: .critical)
                                    }
                                }
                            }
                        }
                    }
                    MinWidthButton("Deactivate License", width: minButtonWidth) {
                        if license?.isAirGapActivated == true {
                            isAirGapDeativationShown = true
                        } else {
                            performAction(name: "Deactivate License") {
                                try license?.deactivate()
                                try manager?.clearLocalStorage()
                            }
                        }
                    }
                    .sheet(isPresented: $isAirGapDeativationShown) {
                        airGapDeactivationView { isAirGapDeativationShown = false }
                            .frame(minWidth: 500)
                            .padding()
                    }
                }
                GroupBox(label: Text("License Info")) {
                    ScrollView(showsIndicators: true) {
                        VStack(alignment: .leading, spacing: 8) {
                            ForEach(0..<licenseInfo.count, id: \.self) {
                                KeyValueView(pair: licenseInfo[$0])
                            }
                        }
                    }
                }
                GroupBox(label: Text("Consumption")) {
                    HStack {
                        KeyValueView(pair: ("Total Count", "\(license?.consumptions ?? 0)"))
                        MinWidthButton("Update Consumption", width: minButtonWidth) {
                            performAction(name: "Update Consumption") {
                                try license?.updateConsumptions(difference: consumptionDelta)
                                consumptionDelta = 0
                            }
                        }
                        .disabled(consumptionDelta == 0)
                        Stepper("\(consumptionDelta)", value: $consumptionDelta)
                    }
                    HStack {
                        KeyValueView(pair: ("Max Count", "\(license?.maxComsumptions ?? 0)"))
                        MinWidthButton("Sync Consumption", width: minButtonWidth) {
                            performAction(name: "Sync Consumption") {
                                try license?.syncConsumptions()
                            }
                        }
                    }
                }
                .disabled(license?.licenseType != .consumption)
                GroupBox(label: Text("Floating")) {
                    HStack {
                        VStack(alignment: .leading) {
                            HStack {
                                MinWidthButton("Borrow", width: minButtonWidth) {
                                    performAction(name: "Borrow") {
                                        try license?.borrow(for: TimeInterval(borrowFor * borrowMultiplier))
                                    }
                                }
                                .disabled(borrowFor == 0)
                                Stepper("\(borrowFor)", value: $borrowFor, in: 1...Int.max)
                                    .frame(width: 40, alignment: .trailing)
                                Picker("", selection: $borrowMultiplier) {
                                    Text("Seconds").tag(1)
                                    Text("Minutes").tag(60)
                                    Text("Hours").tag(60 * 60)
                                }
                                .labelsHidden()
                                .frame(width: 100)
                            }
                            MinWidthButton("Release License", width: minButtonWidth) {
                                performAction(name: "Release License") {
                                    try license?.releaseFloatingLicense()
                                }
                            }
                            .disabled(license?.floatingEndDate == nil)
                            HStack {
                                Picker("Select Feature", selection: $selectedFeature) {
                                    ForEach(license?.features ?? []) { feature in
                                        Text(feature.name).tag(feature as LicenseFeature?)
                                    }
                                }
                                .frame(width: minButtonWidth)
                                
                                MinWidthButton("Register Floating Feature", width: minButtonWidth) {
                                    performAction(name: "Register Floating Feature") {
                                        guard let selectedFeature else { return }
                                        try license?.registerFloatingFeature(selectedFeature)
                                    }
                                }
                                .disabled(selectedFeature == nil)
                            }
                        }
                        Spacer()
                    }
                }
                .disabled(license?.isFloating != true)
            }
        }
        .sheet(isPresented: .constant(productDetails != nil), onDismiss: { productDetails = nil }) {
            ProductDetailsView(productDetails: $productDetails)
        }
    }
    
    var licenseInfo: [KeyValue] {
        [
            ("Key", license?.licenseKey),
            ("User", license?.user?.email),
            ("FloatingID", license?.floatingClientId),
            ("Type", license?.licenseType),
            ("Status", license?.state),
            ("Is activated offline", license?.isOfflineActivated),
            ("Is trial", license?.isTrial),
            ("Is valid", license?.isValid),
            ("Is unlicensed copy", license?.isUnlicensedCopy),
            ("Is floating", license?.isFloating),
            ("Is borrowed", license?.isBorrowed),
            ("Floating period", license?.floatingEndDate?.humanReadable() ?? "-"),
            ("Validity period", license?.validityPeriod?.humanReadable() ?? "-"),
            ("Maintenance period", license?.maintenancePeriod?.humanReadable() ?? "-"),
            ("Grace period (hours)", license.flatMap { $0.gracePeriod / 3600 } ?? "-"),
            ("Last checked", license?.lastSyncDate.humanReadable()),
        ]
    }
    
    private func airGapDeactivationView(onFinish: @escaping () -> Void) -> some View {
        AirGapDeactivationView(
            licenseKey: license?.licenseKey ?? "",
            hardwareID: license?.configuration.hardwareID ?? "",
            generateCode: { request in
                var response = ""
                performAction(name: "Air-gap Dectivation", successAlert: false) {
                    response = try license?.generateAirGapDeactivationCode(initializationCode: request) ?? ""
                }
                return response
            },
            confirmCode: { code in
                performAction(name: "Air-gap Dectivation", successAlert: true) {
                    try license?.deactivateAirGap(confirmationCode: code)
                    try manager?.clearLocalStorage()
                    onFinish()
                }
            }
        )
    }
}

struct LicenseGeneralInfoView_Previews: PreviewProvider {
    static var previews: some View {
        LicenseGeneralInfoView(manager: nil, license: nil)
            .frame(minWidth: 300, minHeight: 500)
    }
}
